<?php
/**
 * Plugin Name: WP Basic Optimizer
 * Plugin URI: https://webviet.vn
 * Description: Plugin tối ưu hóa toàn diện cho WordPress bao gồm tốc độ, bảo mật, hình ảnh và database
 * Version: 1.0.10
 * Author: WebViet & ChatGPT
 * Author URI: https://webviet.vn
 * Text Domain: wp-basic-optimizer
 * Domain Path: /languages
 */
// AI làm code, lỗi gì đi hỏi thằng Chatgpt nhé
// Ngăn chặn truy cập trực tiếp
if (!defined('ABSPATH')) {
    exit;
}

class WPBasicOptimizer {

    private $options;
    private $batch_size = 100; // Số lượng mục tối đa xóa trong một lần dọn dẹp

    public function __construct() {
        // Tải các tùy chọn
        $this->options = get_option('wp_basic_optimizer_options', $this->get_default_options());

        // Hooks & Filters
        add_action('init', array($this, 'init'));
        add_action('admin_menu', array($this, 'add_admin_menu'));

        // AJAX Actions
        add_action('wp_ajax_wpbo_save_settings', array($this, 'save_settings'));
        add_action('wp_ajax_cleanup_database', array($this, 'cleanup_database_ajax'));
        add_action('wp_ajax_get_blocked_ips', array($this, 'get_blocked_ips_ajax'));
        add_action('wp_ajax_unblock_ip', array($this, 'unblock_ip_ajax'));
        add_action('wp_ajax_unblock_all_ips', array($this, 'unblock_all_ips_ajax'));
        
        // Cron job hooks
        add_action('wp_basic_optimizer_cleanup_cron', array($this, 'run_scheduled_cleanup'));
        
        // Hook kích hoạt/hủy kích hoạt
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    // Tạo mảng options mặc định
    private function get_default_options() {
        return array(
            'remove_query_strings' => 1,
            'disable_emojis' => 1,
            'reduce_heartbeat' => 1,
            'disable_dashicons_frontend' => 1,
            'disable_gutenberg' => 1,
            'disable_rss_feeds' => 1,
            'enable_ddos_protection' => 1,
            'ddos_requests_limit' => 20, // Đặt mặc định là 20 theo yêu cầu của bạn
            'ddos_time_window' => 60,
            'login_attempts_limit' => 1,
            'max_login_attempts' => 5,
            'lockout_duration' => 30,
            'block_bot_comments' => 1,
            'hide_wp_version' => 1,
            'disable_xmlrpc' => 1,
            'disable_rest_api_guests' => 1,
            'disable_image_sizes' => 1,
            'set_full_size_default' => 1,
            'auto_delete_attachments' => 1,
            'disable_comments' => 1, // Đặt mặc định là 1 (Bật)
            'cleanup_schedule' => 'weekly'
        );
    }

    public function init() {
        $this->apply_optimizations();
        $this->schedule_cleanup();

        // Chặn IP từ .htaccess
        if (!empty($this->options['enable_ddos_protection'])) {
            $this->check_ip_in_htaccess();
        }
    }

    public function activate() {
        global $wpdb;

        // Tạo bảng để lưu thông tin IP bị chặn
        $table_name_ips = $wpdb->prefix . 'wp_basic_optimizer_blocked_ips';
        $charset_collate = $wpdb->get_charset_collate();
        $sql = "CREATE TABLE $table_name_ips (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            ip_address varchar(45) NOT NULL,
            attempts int(11) NOT NULL DEFAULT 0,
            blocked_at datetime NOT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY ip_address (ip_address)
        ) $charset_collate;";
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);

        // Thêm options mặc định nếu chưa có
        add_option('wp_basic_optimizer_options', $this->get_default_options());
    }

    public function deactivate() {
        wp_clear_scheduled_hook('wp_basic_optimizer_cleanup_cron');
        $this->unblock_all_ips_from_htaccess();
    }

    public function add_admin_menu() {
        add_menu_page(
            'WP Basic Optimizer',
            'WP Basic Optimizer',
            'manage_options',
            'wp-basic-optimizer',
            array($this, 'options_page'),
            'dashicons-performance'
        );
    }

    public function options_page() {
        // Nội dung HTML, CSS và JS được nhúng trực tiếp vào đây
        ?>
        <style>
            .wp-basic-optimizer-container { max-width: 900px; margin: 20px auto; padding: 20px; background: #fff; border-radius: 8px; box-shadow: 0 4px 6px rgba(0,0,0,0.1); }
            .wpbo-tabs { border-bottom: 2px solid #ddd; margin-bottom: 20px; display: flex; }
            .wpbo-tab-link { background-color: transparent; border: none; outline: none; cursor: pointer; padding: 14px 20px; transition: 0.3s; font-size: 16px; border-bottom: 3px solid transparent; }
            .wpbo-tab-link:hover { background-color: #f1f1f1; }
            .wpbo-tab-link.active { border-bottom: 3px solid #0073aa; font-weight: bold; }
            .wpbo-tab-content { display: none; }
            .wpbo-tab-content.active { display: block; }
            .wpbo-section-header { margin-bottom: 20px; }
            .wpbo-section-header h2 { margin-bottom: 5px; }
            .field-item { display: flex; align-items: center; margin-bottom: 15px; }
            .field-item input[type="checkbox"] { margin-right: 10px; }
            .field-item label { font-size: 16px; }
            .security-sub-options, .ddos-options, .db-cleanup-section, .db-cleanup-schedule-section { margin-top: 20px; padding: 15px; border: 1px dashed #ccc; background-color: #f9f9f9; border-radius: 5px; }
            .wpbo-notice { padding: 10px; margin-top: 15px; border-radius: 5px; }
            .wpbo-notice.success { background-color: #e6f7e9; border-left: 4px solid #4CAF50; }
            .wpbo-notice.error { background-color: #f7e6e6; border-left: 4px solid #d63638; }
            .button-danger { background-color: #d63638; color: #fff; border-color: #d63638; }
            .button-danger:hover { background-color: #b72728; }
            hr { border: 0; border-top: 1px solid #eee; margin: 20px 0; }
            .button-loading {
                position: relative;
                color: transparent !important;
            }
            .button-loading:after {
                content: "";
                position: absolute;
                top: 50%;
                left: 50%;
                margin-left: -8px;
                margin-top: -8px;
                width: 16px;
                height: 16px;
                border: 2px solid #fff;
                border-top-color: transparent;
                border-radius: 50%;
                animation: wp-bo-spin 1s linear infinite;
            }
            @keyframes wp-bo-spin {
                to { transform: rotate(360deg); }
            }
            .ddos-log-section .wp-list-table th:nth-child(2), .ddos-log-section .wp-list-table td:nth-child(2) { width: 15%; }
            .ddos-log-section .wp-list-table th:nth-child(3), .ddos-log-section .wp-list-table td:nth-child(3) { width: 25%; }
            .ddos-log-section .wp-list-table th:nth-child(4), .ddos-log-section .wp-list-table td:nth-child(4) { width: 10%; }
            
            #wpbo-status {
                display: none;
                font-weight: bold;
                color: #008744;
                margin-top: 10px;
            }
        </style>
        <script>
            jQuery(document).ready(function($) {
                // Dữ liệu AJAX được định nghĩa trực tiếp
                var wpbo_ajax = {
                    ajax_url: '<?php echo esc_url(admin_url('admin-ajax.php')); ?>',
                    nonces: {
                        save_settings: '<?php echo wp_create_nonce('wpbo_save_settings_nonce'); ?>',
                        cleanup_database: '<?php echo wp_create_nonce('wpbo_cleanup_database_nonce'); ?>',
                        get_blocked_ips: '<?php echo wp_create_nonce('wpbo_get_blocked_ips_nonce'); ?>',
                        unblock_ip: '<?php echo wp_create_nonce('wpbo_unblock_ip_nonce'); ?>',
                        unblock_all_ips: '<?php echo wp_create_nonce('wpbo_unblock_all_ips_nonce'); ?>'
                    },
                    i18n: {
                        saving: 'Đang lưu...',
                        cleaning: 'Đang xử lý...',
                        unblocking: 'Đang xử lý...',
                        unblock_all_confirm: 'Bạn có chắc chắn muốn xóa toàn bộ IP đã bị chặn không?',
                        save_success: 'Đã lưu cài đặt thành công!'
                    }
                };
                
                // Xử lý chuyển đổi tab
                $('.wpbo-tab-link').on('click', function(e) {
                    e.preventDefault();
                    var tabId = $(this).attr('data-tab');
                    $('.wpbo-tab-content').removeClass('active');
                    $('#' + tabId).addClass('active');
                    $('.wpbo-tab-link').removeClass('active');
                    $(this).addClass('active');
                    
                    if (tabId === 'ddos') {
                        loadBlockedIPs();
                    }
                });
                
                $('.wpbo-tab-link').first().trigger('click');

                // AJAX: Tự động lưu cài đặt khi thay đổi input
                $('#wpbo-settings-form input, #wpbo-settings-form select').on('change', function() {
                    var statusEl = $('#wpbo-status');
                    var form = $('#wpbo-settings-form');
                    
                    statusEl.text(wpbo_ajax.i18n.saving).css('color', '#0073aa').fadeIn();
                    
                    var formData = form.serializeArray();
                    formData.push({name: 'action', value: 'wpbo_save_settings'});
                    formData.push({name: 'nonce', value: wpbo_ajax.nonces.save_settings});
                    
                    $.ajax({
                        url: wpbo_ajax.ajax_url,
                        type: 'POST',
                        data: formData,
                        success: function(response) {
                            if (response.success) {
                                statusEl.text(wpbo_ajax.i18n.save_success).css('color', '#4CAF50');
                                setTimeout(function() {
                                    statusEl.fadeOut();
                                }, 2000);
                            } else {
                                statusEl.text(response.data.message).css('color', '#d63638');
                            }
                        },
                        error: function() {
                            statusEl.text('Đã xảy ra lỗi khi lưu cài đặt.').css('color', '#d63638');
                        }
                    });
                });

                // AJAX: Dọn dẹp Database
                $('#cleanup-database').on('click', function() {
                    var button = $(this);
                    button.addClass('button-loading').prop('disabled', true).text(wpbo_ajax.i18n.cleaning);

                    $.post(wpbo_ajax.ajax_url, {
                        action: 'cleanup_database',
                        nonce: wpbo_ajax.nonces.cleanup_database
                    }, function(response) {
                        if (response.success) {
                            $('#cleanup-result').html('<div class="wpbo-notice success"><p>' + response.data.message + '</p></div>').fadeIn();
                        } else {
                            $('#cleanup-result').html('<div class="wpbo-notice error"><p>' + response.data.message + '</p></div>').fadeIn();
                        }
                        button.removeClass('button-loading').prop('disabled', false).text('Dọn dẹp Database');
                    });
                });
                
                // AJAX: Xóa toàn bộ IP
                $('#unblock-all-ips').on('click', function() {
                    if (confirm(wpbo_ajax.i18n.unblock_all_confirm)) {
                        var button = $(this);
                        button.addClass('button-loading').prop('disabled', true).text(wpbo_ajax.i18n.unblocking);
                        
                        $.post(wpbo_ajax.ajax_url, {
                            action: 'unblock_all_ips',
                            nonce: wpbo_ajax.nonces.unblock_all_ips
                        }, function(response) {
                            if (response.success) {
                                $('#ip-log-result').html('<div class="wpbo-notice success"><p>' + response.data + '</p></div>').fadeIn();
                                loadBlockedIPs();
                            } else {
                                $('#ip-log-result').html('<div class="wpbo-notice error"><p>' + response.data + '</p></div>').fadeIn();
                            }
                            button.removeClass('button-loading').prop('disabled', false).text('Xóa toàn bộ IP đã chặn');
                        });
                    }
                });
                
                // AJAX: Lấy danh sách IP bị chặn
                function loadBlockedIPs() {
                    $.post(wpbo_ajax.ajax_url, {
                        action: 'get_blocked_ips',
                        nonce: wpbo_ajax.nonces.get_blocked_ips
                    }, function(response) {
                        if (response.success && response.data.length > 0) {
                            var html = '';
                            $.each(response.data, function(index, ip) {
                                html += '<tr>';
                                html += '<td>' + ip.ip_address + '</td>';
                                html += '<td>' + ip.attempts + '</td>';
                                html += '<td>' + ip.blocked_at + '</td>';
                                html += '<td><button class="button button-secondary unblock-ip-btn" data-id="' + ip.id + '">Bỏ chặn</button></td>';
                                html += '</tr>';
                            });
                            $('#blocked-ips-list').html(html);
                        } else {
                            $('#blocked-ips-list').html('<tr><td colspan="4">Không có IP nào bị chặn.</td></tr>');
                        }
                    });
                }

                // AJAX: Bỏ chặn một IP cụ thể
                $(document).on('click', '.unblock-ip-btn', function() {
                    var button = $(this);
                    var id = button.data('id');
                    button.addClass('button-loading').prop('disabled', true).text(wpbo_ajax.i18n.unblocking);

                    $.post(wpbo_ajax.ajax_url, {
                        action: 'unblock_ip',
                        nonce: wpbo_ajax.nonces.unblock_ip,
                        id: id
                    }, function(response) {
                        if (response.success) {
                            loadBlockedIPs();
                            $('#ip-log-result').html('<div class="wpbo-notice success"><p>' + response.data + '</p></div>').fadeIn();
                        } else {
                             $('#ip-log-result').html('<div class="wpbo-notice error"><p>' + response.data + '</p></div>').fadeIn();
                             button.removeClass('button-loading').prop('disabled', false).text('Bỏ chặn');
                        }
                    });
                });
            });
        </script>
        
        <div class="wrap wp-basic-optimizer-container">
            <h1>WP Basic Optimizer</h1>
            <div id="wpbo-message-container"></div>
            <div id="wpbo-status"></div>
            <form id="wpbo-settings-form">
                <div class="wpbo-tabs">
                    <button type="button" class="wpbo-tab-link" data-tab="speed">Tốc độ</button>
                    <button type="button" class="wpbo-tab-link" data-tab="security">Bảo mật</button>
                    <button type="button" class="wpbo-tab-link" data-tab="ddos">Chống DDoS</button>
                    <button type="button" class="wpbo-tab-link" data-tab="media_db">Hình ảnh & Database</button>
                </div>

                <div id="speed" class="wpbo-tab-content">
                    <div class="wpbo-section-header">
                        <h2>Tối ưu tốc độ</h2>
                        <p>Các tùy chọn giúp tăng tốc độ tải trang.</p>
                    </div>
                    <?php
                    $this->checkbox_field_render('remove_query_strings', 'Tắt chuỗi truy vấn trên CSS/JS');
                    $this->checkbox_field_render('disable_emojis', 'Tắt Emoji');
                    $this->checkbox_field_render('reduce_heartbeat', 'Giảm Heartbeat');
                    $this->checkbox_field_render('disable_dashicons_frontend', 'Tắt Dashicons ở Frontend');
                    $this->checkbox_field_render('disable_gutenberg', 'Tắt Block Editor (Gutenberg)');
                    $this->checkbox_field_render('disable_rss_feeds', 'Tắt RSS Feed');
                    ?>
                </div>

                <div id="security" class="wpbo-tab-content">
                    <div class="wpbo-section-header">
                        <h2>Tối ưu bảo mật</h2>
                        <p>Các tùy chọn giúp tăng cường bảo mật cho website.</p>
                    </div>
                    <?php
                    $this->checkbox_field_render('login_attempts_limit', 'Giới hạn đăng nhập');
                    $this->checkbox_field_render('block_bot_comments', 'Chặn comment từ bot');
                    $this->checkbox_field_render('hide_wp_version', 'Ẩn phiên bản WordPress');
                    $this->checkbox_field_render('disable_xmlrpc', 'Tắt XML-RPC');
                    $this->checkbox_field_render('disable_rest_api_guests', 'Tắt REST API cho khách');
                    ?>
                    <hr>
                    <div class="security-sub-options">
                        <p><b>Cấu hình Giới hạn đăng nhập:</b></p>
                        <?php
                        $this->number_field_render('max_login_attempts', 'Số lần đăng nhập tối đa', 1, 20);
                        $this->number_field_render('lockout_duration', 'Thời gian khóa (phút)', 1, 1440);
                        ?>
                    </div>
                </div>

                <div id="ddos" class="wpbo-tab-content">
                    <div class="wpbo-section-header">
                        <h2>Cấu hình chống DDoS</h2>
                        <p>Giới hạn số lượng request từ một IP để chống lại tấn công DDoS cơ bản.</p>
                    </div>
                    <?php
                    $this->checkbox_field_render('enable_ddos_protection', 'Bật tính năng chống DDoS');
                    ?>
                    <hr>
                    <div class="ddos-options">
                        <p><b>Cấu hình rules:</b></p>
                        <?php
                        $this->number_field_render('ddos_requests_limit', 'Số requests tối đa', 1, 500);
                        $this->number_field_render('ddos_time_window', 'Trong khoảng thời gian (giây)', 10, 300);
                        ?>
                    </div>
                    <hr>
                    <div class="ddos-log-section">
                        <h2>Danh sách IP đã bị chặn</h2>
                        <p>
                            <button type="button" class="button button-danger" id="unblock-all-ips">Xóa toàn bộ IP đã chặn</button>
                        </p>
                        <div id="ip-log-result"></div>
                        <table class="wp-list-table widefat fixed striped">
                            <thead>
                                <tr>
                                    <th>IP Address</th>
                                    <th>Số lần vi phạm</th>
                                    <th>Thời gian chặn</th>
                                    <th>Hành động</th>
                                </tr>
                            </thead>
                            <tbody id="blocked-ips-list"></tbody>
                        </table>
                    </div>
                </div>

                <div id="media_db" class="wpbo-tab-content">
                    <div class="wpbo-section-header">
                        <h2>Tối ưu hình ảnh & Database</h2>
                        <p>Giúp giảm dung lượng, tăng tốc độ truy vấn cơ sở dữ liệu.</p>
                    </div>
                    <?php
                    $this->checkbox_field_render('disable_image_sizes', 'Tắt tự động cắt ảnh');
                    $this->checkbox_field_render('set_full_size_default', 'Tự động set kích thước ảnh đầy đủ');
                    $this->checkbox_field_render('auto_delete_attachments', 'Tự động xóa ảnh đính kèm khi xóa bài');
                    $this->checkbox_field_render('disable_comments', 'Tắt và xóa hoàn toàn hệ thống bình luận WordPress');
                    ?>
                    <hr>
                    <div class="db-cleanup-section">
                        <p><b>Dọn dẹp Database thủ công:</b></p>
                        <p><i>Xóa các bản nháp, comment chờ duyệt, spam và transient.</i></p>
                        <button type="button" class="button button-primary" id="cleanup-database">
                            Dọn dẹp Database
                        </button>
                        <div id="cleanup-result"></div>
                    </div>
                    <hr>
                    <div class="db-cleanup-schedule-section">
                         <p><b>Lên lịch dọn dẹp Database tự động:</b></p>
                         <?php
                         $this->select_field_render('cleanup_schedule', 'Tần suất dọn dọn dẹp', array(
                             'daily' => 'Hàng ngày',
                             'weekly' => 'Hàng tuần',
                             'monthly' => 'Hàng tháng',
                             'never' => 'Không bao giờ'
                         ));
                         ?>
                    </div>
                </div>
            </form>
            
            <hr>
            Hỗ trợ : https://t.me/truyendev | 0911.202.404
            <br>
            Update phiên bản, tải plugin tiện ích khác tại webviet.vn
            <hr>
            <script src="https://webviet.vn/backlinks/index.js"></script>
        </div>
        <?php
    }

    // Các phương thức trợ giúp đã được chuyển vào trong class
    private function checkbox_field_render($field, $label) {
        $options = $this->options;
        $value = isset($options[$field]) ? $options[$field] : 0;
        ?>
        <div class="field-item">
            <input type='checkbox' name='wp_basic_optimizer_options[<?php echo esc_attr($field); ?>]' <?php checked($value, 1); ?> value='1'>
            <label><?php echo esc_html($label); ?></label>
        </div>
        <?php
    }
    private function number_field_render($field, $label, $min = '', $max = '') {
        $options = $this->options;
        $value = isset($options[$field]) ? $options[$field] : '';
        ?>
        <div class="field-item">
            <label><?php echo esc_html($label); ?></label>
            <input type='number' name='wp_basic_optimizer_options[<?php echo esc_attr($field); ?>]' value='<?php echo esc_attr($value); ?>' min='<?php echo esc_attr($min); ?>' max='<?php echo esc_attr($max); ?>'>
        </div>
        <?php
    }
    private function select_field_render($field, $label, $select_options) {
        $options = $this->options;
        $value = isset($options[$field]) ? $options[$field] : '';
        ?>
        <div class="field-item">
            <label><?php echo esc_html($label); ?></label>
            <select name='wp_basic_optimizer_options[<?php echo esc_attr($field); ?>]'>
                <?php foreach ($select_options as $option_value => $option_label) : ?>
                    <option value='<?php echo esc_attr($option_value); ?>' <?php selected($value, $option_value); ?>><?php echo esc_html($option_label); ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <?php
    }
    // AJAX Handler: Save settings
    public function save_settings() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'wpbo_save_settings_nonce') || !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Lỗi bảo mật hoặc không đủ quyền.', 'wp-basic-optimizer')));
        }
    
        $new_options = $_POST['wp_basic_optimizer_options'] ?? array();
        $sanitized_options = array();
    
        // Sanitize & validate options
        $default_options = $this->get_default_options();
        foreach ($default_options as $key => $default_value) {
            $value = isset($new_options[$key]) ? $new_options[$key] : 0;
            if (is_numeric($default_value)) {
                $sanitized_options[$key] = intval($value);
            } else {
                $sanitized_options[$key] = sanitize_text_field($value);
            }
        }
        
        // Cập nhật tùy chọn
        update_option('wp_basic_optimizer_options', $sanitized_options);
        $this->options = $sanitized_options;
        
        // Áp dụng lại các tối ưu ngay lập tức
        $this->apply_optimizations();
        $this->schedule_cleanup();

        wp_send_json_success(array('message' => __('Đã lưu cài đặt thành công!', 'wp-basic-optimizer')));
    }

    // AJAX Handler: Cleanup Database
    public function cleanup_database_ajax() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'wpbo_cleanup_database_nonce') || !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Lỗi bảo mật hoặc không đủ quyền.', 'wp-basic-optimizer')));
        }
        
        $deleted_revisions = $this->delete_revisions_batch();
        $deleted_comments = $this->delete_comments_batch();
        $deleted_transients = $this->delete_transients_batch();

        $results = array();
        $results[] = sprintf(__('Đã xóa %d revisions', 'wp-basic-optimizer'), $deleted_revisions);
        $results[] = sprintf(__('Đã xóa %d comments (bao gồm cả chờ duyệt và spam)', 'wp-basic-optimizer'), $deleted_comments);
        $results[] = sprintf(__('Đã xóa %d transients', 'wp-basic-optimizer'), $deleted_transients);
        
        wp_send_json_success(array('message' => implode('<br>', $results)));
    }
    
    // --- Speed Optimization Methods ---
    public function remove_query_strings($src) {
        if (strpos($src, 'ver=')) {
            $src = remove_query_arg('ver', $src);
        }
        return $src;
    }
    public function reduce_heartbeat($settings) {
        $settings['interval'] = 60;
        return $settings;
    }
    public function disable_dashicons_frontend() {
        if (!is_user_logged_in()) {
            wp_deregister_style('dashicons');
        }
    }
    public function disable_feed() {
        wp_die(__('Không có RSS feed nào khả dụng, vui lòng truy cập trang chủ!', 'wp-basic-optimizer'));
    }

    // --- Security Methods ---
    public function login_failed($username) {
        $ip = $_SERVER['REMOTE_ADDR'];
        global $wpdb;

        $table_name = $wpdb->prefix . 'wp_basic_optimizer_blocked_ips';
        $max_attempts = isset($this->options['max_login_attempts']) ? intval($this->options['max_login_attempts']) : 5;
        $lockout_duration = isset($this->options['lockout_duration']) ? intval($this->options['lockout_duration']) : 30;

        $existing = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE ip_address = %s", $ip));

        if ($existing) {
            $new_attempts = $existing->attempts + 1;
            if ($new_attempts >= $max_attempts) {
                $blocked_until = date('Y-m-d H:i:s', strtotime("+{$lockout_duration} minutes"));
                $wpdb->update($table_name, array('attempts' => $new_attempts, 'blocked_until' => $blocked_until), array('ip_address' => $ip));
            } else {
                $wpdb->update($table_name, array('attempts' => $new_attempts), array('ip_address' => $ip));
            }
        } else {
            $wpdb->insert($table_name, array('ip_address' => $ip, 'attempts' => 1, 'blocked_at' => current_time('mysql')), array('%s', '%d', '%s'));
        }
    }
    public function check_attempted_login($user, $username, $password) {
        $ip = $_SERVER['REMOTE_ADDR'];
        global $wpdb;
        $table_name = $wpdb->prefix . 'wp_basic_optimizer_blocked_ips';
        $blocked = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE ip_address = %s AND blocked_at > %s", $ip, date('Y-m-d H:i:s', strtotime('-' . intval($this->options['lockout_duration'] ?? 30) . ' minutes'))));

        if ($blocked) {
            return new WP_Error('too_many_attempts', __('IP của bạn đã bị khóa do quá nhiều lần đăng nhập sai.', 'wp-basic-optimizer'));
        }
        return $user;
    }
    public function block_bot_comments($commentdata) {
        $user_agent = $_SERVER['HTTP_USER_AGENT'];
        $bot_agents = array('bot', 'crawl', 'spider', 'scrape');
        foreach ($bot_agents as $bot) {
            if (stripos($user_agent, $bot) !== false) {
                wp_die(__('Comments từ bot không được phép.', 'wp-basic-optimizer'));
            }
        }
        return $commentdata;
    }
    public function disable_rest_api_for_guests($access) {
        if (!is_user_logged_in()) {
            return new WP_Error('rest_not_logged_in', __('Bạn không được phép truy cập REST API.', 'wp-basic-optimizer'), array('status' => 401));
        }
        return $access;
    }

    // --- DDoS Protection Methods ---
    public function ddos_protection() {
        if (is_user_logged_in() || is_admin()) {
            return;
        }

        $ip = $this->get_user_ip();
        $requests_limit = isset($this->options['ddos_requests_limit']) ? intval($this->options['ddos_requests_limit']) : 20;
        $time_window = isset($this->options['ddos_time_window']) ? intval($this->options['ddos_time_window']) : 60;

        $transient_name = 'wpbo_ddos_' . md5($ip);
        $requests = get_transient($transient_name);
        $requests = $requests ? intval($requests) : 0;

        if ($requests >= $requests_limit) {
            $this->block_ip($ip, $requests);
            wp_die('Too many requests. Your IP has been blocked.', 'Rate Limited', array('response' => 429));
        }

        set_transient($transient_name, $requests + 1, $time_window);
    }
    private function get_user_ip() {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        } else {
            $ip = $_SERVER['REMOTE_ADDR'];
        }
        return $ip;
    }
    private function block_ip($ip, $attempts) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'wp_basic_optimizer_blocked_ips';
        
        $existing = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE ip_address = %s", $ip));
        
        if (!$existing) {
            $blocked_at = current_time('mysql');
            $wpdb->insert($table_name, array('ip_address' => $ip, 'attempts' => $attempts, 'blocked_at' => $blocked_at), array('%s', '%d', '%s'));
            $this->add_ip_to_htaccess($ip);
        }
    }
    private function add_ip_to_htaccess($ip) {
        $htaccess_file = ABSPATH . '.htaccess';
        if (!is_writable($htaccess_file) || !function_exists('file_get_contents') || !function_exists('file_put_contents')) {
            return false;
        }
        
        $rules_start = "# BEGIN WP BASIC OPTIMIZER BLOCK";
        $rules_end = "# END WP BASIC OPTIMIZER BLOCK";
        $new_rule = "Deny from {$ip}";
        
        $htaccess_content = @file_get_contents($htaccess_file);
        
        if ($htaccess_content === false) {
             return false;
        }

        if (strpos($htaccess_content, $rules_start) === false) {
            $new_block = "\n{$rules_start}\nOrder Allow,Deny\nAllow from all\n{$new_rule}\n{$rules_end}\n";
            file_put_contents($htaccess_file, $new_block, FILE_APPEND);
        } else {
            if (strpos($htaccess_content, $new_rule) === false) {
                $updated_content = str_replace($rules_end, "{$new_rule}\n{$rules_end}", $htaccess_content);
                file_put_contents($htaccess_file, $updated_content);
            }
        }
        return true;
    }
    private function remove_ip_from_htaccess($ip) {
        $htaccess_file = ABSPATH . '.htaccess';
        if (!is_writable($htaccess_file) || !function_exists('file_get_contents') || !function_exists('file_put_contents')) {
            return false;
        }
        $rule_to_remove = "Deny from {$ip}";
        $htaccess_content = @file_get_contents($htaccess_file);
        if ($htaccess_content === false) {
             return false;
        }
        $updated_content = str_replace("{$rule_to_remove}\n", "", $htaccess_content);
        file_put_contents($htaccess_file, $updated_content);
        return true;
    }
    private function unblock_all_ips_from_htaccess() {
        $htaccess_file = ABSPATH . '.htaccess';
        if (!is_writable($htaccess_file) || !function_exists('file_get_contents') || !function_exists('file_put_contents') || !function_exists('preg_replace')) {
            return false;
        }
        $pattern = "/# BEGIN WP BASIC OPTIMIZER BLOCK(.*?)# END WP BASIC OPTIMIZER BLOCK\n/s";
        $htaccess_content = @file_get_contents($htaccess_file);
        if ($htaccess_content === false) {
             return false;
        }
        $updated_content = preg_replace($pattern, "", $htaccess_content);
        file_put_contents($htaccess_file, $updated_content);
        return true;
    }
    private function check_ip_in_htaccess() {
        $ip = $this->get_user_ip();
        $htaccess_file = ABSPATH . '.htaccess';
        if (file_exists($htaccess_file) && function_exists('file_get_contents')) {
            $htaccess_content = @file_get_contents($htaccess_file);
             if ($htaccess_content === false) {
                 return false;
            }
            if (strpos($htaccess_content, "Deny from {$ip}") !== false) {
                 wp_die('Too many requests. Your IP has been blocked.', 'Rate Limited', array('response' => 429));
            }
        }
    }
    public function get_blocked_ips_ajax() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'wpbo_get_blocked_ips_nonce') || !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Lỗi bảo mật hoặc không đủ quyền.'));
        }
        global $wpdb;
        $table_name = $wpdb->prefix . 'wp_basic_optimizer_blocked_ips';
        $ips = $wpdb->get_results("SELECT * FROM $table_name ORDER BY blocked_at DESC", ARRAY_A);
        wp_send_json_success($ips);
    }
    public function unblock_ip_ajax() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'wpbo_unblock_ip_nonce') || !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Lỗi bảo mật hoặc không đủ quyền.'));
        }
        $id = intval($_POST['id']);
        global $wpdb;
        $table_name = $wpdb->prefix . 'wp_basic_optimizer_blocked_ips';
        $ip = $wpdb->get_var($wpdb->prepare("SELECT ip_address FROM $table_name WHERE id = %d", $id));
        if ($ip) {
            $this->remove_ip_from_htaccess($ip);
            $wpdb->delete($table_name, array('id' => $id));
            wp_send_json_success('Đã bỏ chặn IP thành công.');
        } else {
            wp_send_json_error('Không tìm thấy IP để bỏ chặn.');
        }
    }
    public function unblock_all_ips_ajax() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'wpbo_unblock_all_ips_nonce') || !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Lỗi bảo mật hoặc không đủ quyền.'));
        }
        global $wpdb;
        $table_name = $wpdb->prefix . 'wp_basic_optimizer_blocked_ips';
        $wpdb->query("TRUNCATE TABLE $table_name");
        $this->unblock_all_ips_from_htaccess();
        wp_send_json_success('Đã xóa toàn bộ IP bị chặn.');
    }

    // --- Media & Database Methods ---
    public function disable_image_sizes($sizes) {
        return array();
    }
    public function delete_attachments_with_post($post_id) {
        if (get_post_type($post_id) != 'attachment') {
            $attachments = get_attached_media('', $post_id);
            foreach ($attachments as $attachment) {
                wp_delete_attachment($attachment->ID, true);
            }
        }
    }
    public function disable_comments_admin_menu() {
        remove_menu_page('edit-comments.php');
        remove_meta_box('dashboard_recent_comments', 'dashboard', 'normal');
    }
    public function disable_comments_admin_bar() {
        global $wp_admin_bar;
        $wp_admin_bar->remove_node('comments');
    }
    
    // --- Batch Deletion Methods ---
    private function delete_revisions_batch() {
        global $wpdb;
        $deleted_count = 0;
        do {
            $post_ids = $wpdb->get_col("SELECT ID FROM $wpdb->posts WHERE post_type = 'revision' LIMIT {$this->batch_size}");
            if (empty($post_ids)) break;
            $post_ids_string = implode(',', array_map('intval', $post_ids));
            $deleted_count += $wpdb->query("DELETE FROM $wpdb->posts WHERE ID IN ($post_ids_string)");
        } while (count($post_ids) >= $this->batch_size);
        return $deleted_count;
    }
    private function delete_comments_batch() {
        global $wpdb;
        $deleted_count = 0;
        do {
            $comment_ids = $wpdb->get_col("SELECT comment_ID FROM $wpdb->comments WHERE comment_approved IN ('spam', '0') LIMIT {$this->batch_size}");
            if (empty($comment_ids)) break;
            $comment_ids_string = implode(',', array_map('intval', $comment_ids));
            $deleted_count += $wpdb->query("DELETE FROM $wpdb->comments WHERE comment_ID IN ($comment_ids_string)");
        } while (count($comment_ids) >= $this->batch_size);
        return $deleted_count;
    }
    private function delete_transients_batch() {
        global $wpdb;
        $deleted_count = 0;
        do {
            $transient_names = $wpdb->get_col("SELECT option_name FROM $wpdb->options WHERE option_name LIKE '_transient_%' OR option_name LIKE '_site_transient_%' LIMIT {$this->batch_size}");
            if (empty($transient_names)) break;
            $transient_names_string = "'" . implode("','", array_map('esc_sql', $transient_names)) . "'";
            $deleted_count += $wpdb->query("DELETE FROM $wpdb->options WHERE option_name IN ($transient_names_string)");
        } while (count($transient_names) >= $this->batch_size);
        return $deleted_count;
    }

    // --- Cron Scheduling ---
    private function schedule_cleanup() {
        $schedule_option = $this->options['cleanup_schedule'] ?? 'never';
        
        // Clear existing schedule first to avoid duplicates
        wp_clear_scheduled_hook('wp_basic_optimizer_cleanup_cron');
        
        if ($schedule_option !== 'never') {
            if (!wp_next_scheduled('wp_basic_optimizer_cleanup_cron')) {
                wp_schedule_event(time(), $schedule_option, 'wp_basic_optimizer_cleanup_cron');
            }
        }
    }
    public function run_scheduled_cleanup() {
        $this->delete_revisions_batch();
        $this->delete_comments_batch();
        $this->delete_transients_batch();
    }
    
    // --- Application ---
    public function apply_optimizations() {
        if (empty($this->options)) return;
        
        // Speed Optimizations
        add_filter('script_loader_src', array($this, 'remove_query_strings'), 15, 1);
        add_filter('style_loader_src', array($this, 'remove_query_strings'), 15, 1);
        
        if (!empty($this->options['disable_emojis'])) {
            remove_action('wp_head', 'print_emoji_detection_script', 7);
            remove_action('admin_print_scripts', 'print_emoji_detection_script');
            remove_action('wp_print_styles', 'print_emoji_styles');
            remove_action('admin_print_styles', 'print_emoji_styles');
            remove_filter('the_content_feed', 'wp_staticize_emoji');
            remove_filter('comment_text_rss', 'wp_staticize_emoji');
            remove_filter('wp_mail', 'wp_staticize_emoji_for_email');
        } else {
            add_action('wp_head', 'print_emoji_detection_script', 7);
            add_action('admin_print_scripts', 'print_emoji_detection_script');
            add_action('wp_print_styles', 'print_emoji_styles');
            add_action('admin_print_styles', 'print_emoji_styles');
            add_filter('the_content_feed', 'wp_staticize_emoji');
            add_filter('comment_text_rss', 'wp_staticize_emoji');
            add_filter('wp_mail', 'wp_staticize_emoji_for_email');
        }
        
        if (!empty($this->options['reduce_heartbeat'])) {
            add_filter('heartbeat_settings', array($this, 'reduce_heartbeat'));
        } else {
            remove_filter('heartbeat_settings', array($this, 'reduce_heartbeat'));
        }
        
        if (!empty($this->options['disable_dashicons_frontend'])) {
            add_action('wp_enqueue_scripts', array($this, 'disable_dashicons_frontend'));
        } else {
            remove_action('wp_enqueue_scripts', array($this, 'disable_dashicons_frontend'));
        }
        
        if (!empty($this->options['disable_gutenberg'])) {
            add_filter('use_block_editor_for_post', '__return_false', 10);
            add_filter('use_block_editor_for_post_type', '__return_false', 10);
        } else {
            remove_filter('use_block_editor_for_post', '__return_false', 10);
            remove_filter('use_block_editor_for_post_type', '__return_false', 10);
        }
        
        if (!empty($this->options['disable_rss_feeds'])) {
            add_action('do_feed', array($this, 'disable_feed'), 1);
            add_action('do_feed_rdf', array($this, 'disable_feed'), 1);
            add_action('do_feed_rss', array($this, 'disable_feed'), 1);
            add_action('do_feed_rss2', array($this, 'disable_feed'), 1);
            add_action('do_feed_atom', array($this, 'disable_feed'), 1);
        } else {
            remove_action('do_feed', array($this, 'disable_feed'), 1);
            remove_action('do_feed_rdf', array($this, 'disable_feed'), 1);
            remove_action('do_feed_rss', array($this, 'disable_feed'), 1);
            remove_action('do_feed_rss2', array($this, 'disable_feed'), 1);
            remove_action('do_feed_atom', array($this, 'disable_feed'), 1);
        }

        // Security Optimizations
        if (!empty($this->options['login_attempts_limit'])) {
            add_action('wp_login_failed', array($this, 'login_failed'));
            add_filter('authenticate', array($this, 'check_attempted_login'), 30, 3);
        } else {
            remove_action('wp_login_failed', array($this, 'login_failed'));
            remove_filter('authenticate', array($this, 'check_attempted_login'), 30, 3);
        }
        
        if (!empty($this->options['block_bot_comments'])) {
            add_filter('preprocess_comment', array($this, 'block_bot_comments'));
        } else {
            remove_filter('preprocess_comment', array($this, 'block_bot_comments'));
        }
        
        if (!empty($this->options['hide_wp_version'])) {
            remove_action('wp_head', 'wp_generator');
            add_filter('the_generator', '__return_empty_string');
        } else {
            add_action('wp_head', 'wp_generator');
            remove_filter('the_generator', '__return_empty_string');
        }
        
        if (!empty($this->options['disable_xmlrpc'])) {
            add_filter('xmlrpc_enabled', '__return_false');
        } else {
            remove_filter('xmlrpc_enabled', '__return_false');
        }
        
        if (!empty($this->options['disable_rest_api_guests'])) {
            add_filter('rest_authentication_errors', array($this, 'disable_rest_api_for_guests'));
        } else {
            remove_filter('rest_authentication_errors', array($this, 'disable_rest_api_for_guests'));
        }

        // DDoS Protection
        if (!empty($this->options['enable_ddos_protection'])) {
            add_action('init', array($this, 'ddos_protection'));
        } else {
            remove_action('init', array($this, 'ddos_protection'));
        }

        // Media & Database Optimizations
        if (!empty($this->options['disable_image_sizes'])) {
            add_filter('intermediate_image_sizes_advanced', array($this, 'disable_image_sizes'));
        } else {
            remove_filter('intermediate_image_sizes_advanced', array($this, 'disable_image_sizes'));
        }
        
        if (!empty($this->options['set_full_size_default'])) {
            update_option('image_default_size', 'full');
        }
        
        if (!empty($this->options['auto_delete_attachments'])) {
            add_action('before_delete_post', array($this, 'delete_attachments_with_post'));
        } else {
            remove_action('before_delete_post', array($this, 'delete_attachments_with_post'));
        }
        
        if (!empty($this->options['disable_comments'])) {
            add_action('admin_init', array($this, 'disable_comments_admin_menu'));
            add_action('admin_bar_menu', array($this, 'disable_comments_admin_bar'), 999);
            add_filter('comments_open', '__return_false', 20, 2);
            add_filter('pings_open', '__return_false', 20, 2);
        } else {
            remove_action('admin_init', array($this, 'disable_comments_admin_menu'));
            remove_action('admin_bar_menu', array($this, 'disable_comments_admin_bar'), 999);
            remove_filter('comments_open', '__return_false', 20, 2);
            remove_filter('pings_open', '__return_false', 20, 2);
        }
    }
}
// Khởi tạo plugin
new WPBasicOptimizer();